<?php

namespace App\Exceptions;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\ValidationException;
use Throwable;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Symfony\Component\HttpKernel\Exception\HttpException;
use WpOrg\Requests\Proxy\Http;

class Handler extends ExceptionHandler
{
    /**
     * A list of the inputs that are never flashed for validation exceptions.
     *
     * @var array
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     */
    public function register(): void
    {
        $this->reportable(function (Throwable $e) {
            //
        });
    }

    public function render($request, Throwable $exception)
    {
        $code = $exception->getCode();
        $message = $exception->getMessage();
        if ($code < 100 || $code >= 600) {
            $code = \Illuminate\Http\Response::HTTP_INTERNAL_SERVER_ERROR;
        }

        if ($exception instanceof ModelNotFoundException) {
            $message = $exception->getMessage();
            $code = \Illuminate\Http\Response::HTTP_NOT_FOUND;

            if (preg_match('@\\\\(\w+)\]@', $message, $matches)) {
                $model = $matches[1];
                $model = preg_replace('/Table/i', '', $model);
                $message = "{$model} not found.";
            }
        }

        if ($exception instanceof ValidationException) {
            $validator = $exception->validator;
            $message = $validator->errors()->first();
            $code = \Illuminate\Http\Response::HTTP_UNPROCESSABLE_ENTITY;

            if (! $request->expectsJson() and ! $request->isXmlHttpRequest()) {
                return Redirect::back()->withInput()->withErrors($message);
            }
        }

        if ($request->expectsJson() or $request->isXmlHttpRequest()) {
            return Response::json([
                'success' => false,
                'message' => $message,
            ], $code);
        }
        if ($exception instanceof NotFoundHttpException){
            Log::channel('404log')->info('404 Error Ocurred',[
                'url' => $request->fullUrl(),
                'ip' => $request->ip(),
                'user_agent'=> $request->header('user-agent'),
                 'time' => now()->toDateTimeString(),
            ]);
        }
        // 404 ERROR LOGGING
        if($exception instanceof NotFoundHttpException) {
           $guid = (string) \Illuminate\Support\Str::uuid();

            $path = storage_path("logs/404/{$guid}.log");

            $logger = new \Monolog\Logger('404_guid_logger');
            $logger->pushHandler(new \Monolog\Handler\StreamHandler($path));

            $logger->info('404 Error Occurred', [
                'guid' => $guid,
                'url' => $request->fullUrl(),
                'ip' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
                'time' => now()->toDateTimeString(),
            ]);

        }
        // 500 ERROR LOGGGING
        if (
            !($exception instanceof NotFoundHttpException) &&   // not 404
            !($exception instanceof ValidationException) &&      // not validation error
            !($exception instanceof HttpException && $exception->getStatusCode() < 500) // not 400/401/403/429 etc
        ) {
            $guid = (string) Str::uuid();
            $path = storage_path("logs/500/{$guid}.log");

            $logger = new \Monolog\Logger('500_guid_logger');
            $logger->pushHandler(new \Monolog\Handler\StreamHandler($path));

            $logger->error('500 Internal Server Error', [
                'guid' => $guid,
                'url' => $request->fullUrl(),
                'ip' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
                'message' => $exception->getMessage(),
                'time' => now()->toDateTimeString(),
            ]);
        }
        // 403 LOGGING
        if ($this->isHttpException($exception) && $exception->getStatusCode() == 403) {
            // Generate GUID
            $guid = (string) Str::uuid();

            // Create log file name
            $fileName = "403-{$guid}.log";

            // What to log
            $data = [
                'guid' => $guid,
                'url'   => $request->fullUrl(),
                'ip'    => $request->ip(),
                'method' => $request->method(),
                'user_id' => auth()->id(),
                'timestamp' => now()->toDateTimeString(),
                'message' => $exception->getMessage(),
            ];
            // Write to storage/logs/*.log
            Log::build([
                'driver' => 'single',
                'path' => storage_path("logs/{$fileName}")
            ])->info('403 Error Occurred', $data);
        }
        // 401 LOGGING
        if ($this->isHttpException($exception) && $exception->getStatusCode() == 401) {

            // Generate GUID
            $guid = (string) Str::uuid();

            // File name
            $fileName = "401-{$guid}.log";

            // Log data
            $data = [
                'guid' => $guid,
                'url'   => $request->fullUrl(),
                'ip'    => $request->ip(),
                'method' => $request->method(),
                'user_id' => auth()->id(),
                'timestamp' => now()->toDateTimeString(),
                'message' => $exception->getMessage(),
            ];

            // Write file to storage/logs/
            Log::build([
                'driver' => 'single',
                'path' => storage_path("logs/{$fileName}")
            ])->info('401 Error Occurred', $data);
        }
        // 419 LOGGING
        if (
            $exception instanceof \Illuminate\Session\TokenMismatchException ||
            ($this->isHttpException($exception) && $exception->getStatusCode() == 419)
        ) {

            // Generate GUID
            $guid = (string) Str::uuid();

            // File name
            $fileName = "419-{$guid}.log";

            // Log content
            $data = [
                'guid' => $guid,
                'url' => $request->fullUrl(),
                'ip' => $request->ip(),
                'method' => $request->method(),
                'user_agent' => $request->header('User-Agent'),
                'user_id' => auth()->id(),
                'timestamp' => now()->toDateTimeString(),
                'message' => $exception->getMessage() ?: 'CSRF Token Mismatch / Page Expired'
            ];

            // Create log file in storage/logs/
            Log::build([
                'driver' => 'single',
                'path' => storage_path("logs/{$fileName}")
            ])->error('419 Page Expired', $data);
        }
        // 429 LOGGING
        if (
            $exception instanceof \Illuminate\Http\Exceptions\ThrottleRequestsException ||
            ($this->isHttpException($exception) && $exception->getStatusCode() == 429)
        ) {

            $guid = (string) Str::uuid();
            $fileName = "429-{$guid}.log";

            $data = [
                'guid' => $guid,
                'url' => $request->fullUrl(),
                'ip' => $request->ip(),
                'method' => $request->method(),
                'user_agent' => $request->header('User-Agent'),
                'user_id' => auth()->id(),
                'timestamp' => now()->toDateTimeString(),
                'message' => $exception->getMessage() ?: 'Too Many Requests / Rate Limit Exceeded'
            ];

            Log::build([
                'driver' => 'single',
                'path' => storage_path("logs/{$fileName}")
            ])->error('429 Too Many Requests', $data);
        }
        // 503 LOGGING
        if ($this->isHttpException($exception) && $exception->getStatusCode() == 503) {

            $guid = (string) Str::uuid();
            $fileName = "503-{$guid}.log";

            $data = [
                'guid' => $guid,
                'url' => $request->fullUrl(),
                'ip' => $request->ip(),
                'method' => $request->method(),
                'user_agent' => $request->header('User-Agent'),
                'user_id' => auth()->id(),
                'timestamp' => now()->toDateTimeString(),
                'message' => $exception->getMessage() ?: 'Service Unavailable / Maintenance Mode',
            ];

            Log::build([
                'driver' => 'single',
                'path' => storage_path("logs/{$fileName}")
            ])->error('503 Service Unavailable', $data);
        }
        return parent::render($request, $exception);
    }
}
