<?php

namespace App\Http\Controllers;

use App\Http\Requests\createWithdrawAmountRequest;
use App\Jobs\SendWithdrawRequestMailJob;
use App\Mail\SendinviteMail;
use App\Models\AffiliateUser;
use App\Models\Withdrawal;
use App\Models\WithdrawalTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\URL;
use Illuminate\View\View;

class AffiliationWithdrawController extends AppBaseController
{
    public function affiliateWithdraw()
    {
        $currentPlan = getCurrentSubscription()->plan;

        if (! $currentPlan->planFeature->affiliation) {
            return redirect()->route('admin.dashboard');
        }

        $currentUserId = Auth::id();

        // Get minimum withdrawal amount based on user's current plan currency
        $currentSubscription = getCurrentSubscription();
        $currentSubscription->load('plan.currency');
        $planCurrencyCode = $currentSubscription->plan->currency->currency_code ?? 'USD';

        if ($planCurrencyCode === 'LKR') {
            $minimumWithdrawalAmount = getSuperAdminSettingValue('minimum_withdrawal_amount_lkr') ?? 1000;
            $currencySymbol = 'LKR';
            $affiliationAmount = getSuperAdminSettingValue('affiliation_amount_lkr') ?? 1000;
        } else {
            $minimumWithdrawalAmount = getSuperAdminSettingValue('minimum_withdrawal_amount_usd') ?? 10;
            $currencySymbol = 'USD';
            $affiliationAmount = getSuperAdminSettingValue('affiliation_amount') ?? 10;
        }

        // Calculate amounts for the user's current plan currency
        $totalAmount = AffiliateUser::whereAffiliatedBy($currentUserId)
            ->where('currency_code', $planCurrencyCode)
            ->sum('amount');

        // Also get total for other currency to show if there are mixed currencies
        $otherCurrencyCode = ($planCurrencyCode === 'LKR') ? 'USD' : 'LKR';
        $totalAmountOther = AffiliateUser::whereAffiliatedBy($currentUserId)
            ->where('currency_code', $otherCurrencyCode)
            ->sum('amount');

        $currentAmount = currentAffiliateAmount($currentUserId, $planCurrencyCode);

        // If there are amounts in other currency, we need to handle this
        // For now, we'll only show amounts in the user's current currency
        // Note: This means if user switches plans, they'll see different totals

        return view('user-settings.affiliationWithdraw.index', compact('totalAmount', 'currentAmount', 'minimumWithdrawalAmount', 'currencySymbol', 'affiliationAmount'));
    }

    public function withdrawAmount(createWithdrawAmountRequest $request)
    {
        $currentUserId = getLogInUserId();
        $inProcessWithdrawal = Withdrawal::whereUserId($currentUserId)->whereIsApproved(Withdrawal::INPROCESS)->first();
        if ($inProcessWithdrawal) {
            return $this->sendError('Your Last Withdraw Request is pending, you can not add another request.');
        }
        $input = $request->all();

        // Check minimum withdrawal amount based on user's current plan currency
        $currentSubscription = getCurrentSubscription();
        $currentSubscription->load('plan.currency');
        $planCurrencyCode = $currentSubscription->plan->currency->currency_code ?? 'USD';

        if ($input['amount'] > currentAffiliateAmount($currentUserId, $planCurrencyCode)) {
            return $this->sendError('Your can not withdraw more than your current balance');
        }

        if ($planCurrencyCode === 'LKR') {
            $minimumAmount = getSuperAdminSettingValue('minimum_withdrawal_amount_lkr') ?? 1000;
            $currencyIcon = 'LKR';
        } else {
            $minimumAmount = getSuperAdminSettingValue('minimum_withdrawal_amount_usd') ?? 10;
            $currencyIcon = '$';
        }

        // Format amounts with correct currency symbol
        $currencyPosition = getSuperAdminSettingValue('currency_after_amount');
        $formattedMinimumAmount = $currencyPosition
            ? number_format($minimumAmount, 2) . $currencyIcon
            : $currencyIcon . number_format($minimumAmount, 2);

        if ($input['amount'] < $minimumAmount) {
            return $this->sendError("Minimum withdrawal amount is {$formattedMinimumAmount}. Your current amount must be at least this amount to withdraw.");
        }

        // Check if current amount is sufficient
        $currentAmount = currentAffiliateAmount($currentUserId, $planCurrencyCode);
        $formattedCurrentAmount = $currencyPosition
            ? number_format($currentAmount, 2) . $currencyIcon
            : $currencyIcon . number_format($currentAmount, 2);
        $formattedMinimumAmountForError = $currencyPosition
            ? number_format($minimumAmount, 2) . $currencyIcon
            : $currencyIcon . number_format($minimumAmount, 2);

        if ($currentAmount < $minimumAmount) {
            return $this->sendError("Your current amount ({$formattedCurrentAmount}) is less than the minimum withdrawal amount ({$formattedMinimumAmountForError}).");
        }

        // Get paypal_email from input or use stored user setting
        $paypalEmail = $input['paypal_email'] ?? getUserSettingValue('paypal_email', $currentUserId);

        // If paypal_email is provided in the request, validate it matches stored email
        if (isset($input['paypal_email']) && !empty($input['paypal_email'])) {
            if (getUserSettingValue('paypal_email', $currentUserId) != $input['paypal_email']) {
                return $this->sendError('The Email does not match with your PayPal email.');
            }
        }

        $withdrawal = new Withdrawal();
        $withdrawal->amount = $input['amount'];
        $withdrawal->user_id = $currentUserId;
        $withdrawal->email = $paypalEmail ?? ''; // Use paypal_email from input or stored setting, or empty string
        $withdrawal->is_approved = Withdrawal::INPROCESS;
        $withdrawal->save();

        return $this->sendResponse($withdrawal, 'Withdraw request sent successfully.');
    }

    public function affiliationWithdraw(): View
    {

        return view('sadmin.affiliationWithdraw.index');
    }

    public function changeWithdrawalStatus(Request $request, $id, $approval)
    {
        $rejectionNote = $request->rejectionNote;
        $meta = $request->meta;
        $withdrawal = Withdrawal::find($id);
        $withdrawal->update([
            'is_approved' => $approval,
            'rejection_note' => ($approval == Withdrawal::REJECTED) ? $rejectionNote : null,
        ]);

        if ($approval == Withdrawal::APPROVED) {
            WithdrawalTransaction::create([
                'withdrawal_id' => $withdrawal->id,
                'amount' => $withdrawal->amount,
                'paid_by' => $meta ? WithdrawalTransaction::PAYPAL : WithdrawalTransaction::MANUALLY,
                'payment_meta' => $meta,
            ]);
        }

        SendWithdrawRequestMailJob::dispatch($withdrawal->id, $approval);

        return $this->sendResponse($withdrawal, 'Withdrawal status updated successfully');
    }

    public function withdrawTransaction(): View
    {
        return view('sadmin.withdrawalTransactions.index');
    }

    public function showAffiliationWithdraw($id)
    {
        $affiliationWithdraw = Withdrawal::with('user')->find($id);
        $affiliationWithdraw['formattedAmount'] = currencyFormat($affiliationWithdraw->amount, 2);

        if (isAdmin() && $affiliationWithdraw->user_id !== getLogInUserId()) {
            return $this->sendError('Withdrawal data not found');
        }

        return $this->sendResponse($affiliationWithdraw, 'Withdrawal data retrieved successfully.');
    }

    public function sendInvite(Request $request)
    {
        $affiliateCode = getLogInUser()->affiliate_code;
        $affliateName = getLogInUser()->full_name;

        $referralURL = URL::to('/register?referral-code='.$affiliateCode);
        $input = [
            'referralUrl' => $referralURL,
            'username' => $affliateName,
        ];
        Mail::to($request['email'])
            ->send(new SendinviteMail($input, $request['email']));

        return $this->sendSuccess('Successfully send');
    }
}
