<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\CreateRegisterRequest;
use App\Mail\VerifyMail;
use App\Models\AffiliateUser;
use App\Models\MultiTenant;
use App\Models\Plan;
use App\Models\Role;
use App\Models\Setting;
use App\Models\Subscription;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\RedirectResponse;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Password;
use Laracasts\Flash\Flash;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;

class RegisteredUserController extends Controller
{
    /**
     * @return Application|Factory|\Illuminate\Contracts\View\View
     */
    public function create()
    {
        if (! getSuperAdminSettingValue('register_enable')) {
            return redirect()->back();
        }

        return view('auth.register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @return Application|RedirectResponse|Redirector
     */
    public function store(CreateRegisterRequest $request): RedirectResponse
    {
        $referral_code = $request->input('referral-code');
        $referral_user = '';
        if ($referral_code) {
            $referral_user = User::where('affiliate_code', $referral_code)->first();
        }
        try {
            DB::beginTransaction();

            $tenant = MultiTenant::create(['tenant_username' => $request->first_name]);
            $userDefaultLanguage = Setting::where('key', 'user_default_language')->first()->value ?? 'en';

            $user = User::create([
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'language' => $userDefaultLanguage,
                'password' => Hash::make($request->password),
                'tenant_id' => $tenant->id,
                'affiliate_code' => generateUniqueAffiliateCode(),
                'currency_preference' => $request->currency_preference ?? 'USD',
            ])->assignRole(Role::ROLE_ADMIN);

            // Select the default plan based on user's currency preference
            $userCurrencyPreference = $request->input('currency_preference', 'USD');
            // Handle backward compatibility: convert 'default' to 'USD'
            if ($userCurrencyPreference === 'default') {
                $userCurrencyPreference = 'USD';
            }

            // Get default plan matching the currency preference
            $planQuery = Plan::with('currency')->whereIsDefault(true);

            if ($userCurrencyPreference === 'LKR') {
                // Get default LKR plan
                $planQuery->whereHas('currency', function($q) {
                    $q->where('currency_code', 'LKR');
                });
            } else {
                // Get default non-LKR plan (USD, EUR, etc.)
                $planQuery->whereHas('currency', function($q) {
                    $q->where('currency_code', '!=', 'LKR');
                });
            }

            $plan = $planQuery->first();

            // Fallback to any default plan if no match found
            if (!$plan) {
                $plan = Plan::with('currency')->whereIsDefault(true)->first();
            }

            Subscription::create([
                'plan_id' => $plan->id,
                'plan_amount' => $plan->price,
                'plan_frequency' => Plan::MONTHLY,
                'starts_at' => Carbon::now(),
                'ends_at' => Carbon::now()->addDays($plan->trial_days),
                'trial_ends_at' => Carbon::now()->addDays($plan->trial_days),
                'status' => Subscription::ACTIVE,
                'tenant_id' => $tenant->id,
                'no_of_vcards' => $plan->no_of_vcards,
            ]);

            if ($referral_user) {
                // Get affiliation amount based on the REFERRAL USER's plan currency
                // The referrer's plan currency determines what affiliation amount the new user should receive
                $referralUserSubscription = Subscription::with('plan.currency')
                    ->where('tenant_id', $referral_user->tenant_id)
                    ->where('status', Subscription::ACTIVE)
                    ->orderBy('created_at', 'desc')
                    ->first();

                // Get the referral user's plan currency
                $referralPlanCurrencyCode = 'USD'; // Default fallback
                if ($referralUserSubscription && $referralUserSubscription->plan && $referralUserSubscription->plan->currency) {
                    $referralPlanCurrencyCode = $referralUserSubscription->plan->currency->currency_code;
                }

                // Determine affiliation amount based on referrer's plan currency
                if ($referralPlanCurrencyCode === 'LKR') {
                    $affiliationAmount = getSuperAdminSettingValue('affiliation_amount_lkr') ?? 1000;
                } else {
                    $affiliationAmount = getSuperAdminSettingValue('affiliation_amount') ?? 10;
                }

                // Debug: Log the values to help diagnose issues
                Log::info('Affiliate Registration - Referrer Plan Currency Based', [
                    'new_user_currency_preference' => $userCurrencyPreference,
                    'new_user_plan_id' => $plan->id,
                    'new_user_plan_currency_code' => $plan->currency->currency_code ?? 'USD',
                    'referral_user_id' => $referral_user->id,
                    'referral_user_plan_id' => $referralUserSubscription->plan->id ?? null,
                    'referral_user_plan_currency_code' => $referralPlanCurrencyCode,
                    'affiliation_currency_code' => $referralPlanCurrencyCode,
                    'affiliation_amount' => $affiliationAmount,
                    'new_user_id' => $user->id,
                ]);

                $affiliateUser = new AffiliateUser();
                $affiliateUser->affiliated_by = $referral_user->id;
                $affiliateUser->user_id = $user->id;
                $affiliateUser->amount = $affiliationAmount;
                $affiliateUser->currency_code = $referralPlanCurrencyCode; // Use referrer's plan currency
                $affiliateUser->is_verified = true; // Ensure this is set
                $affiliateUser->save();
            }

            DB::commit();
            $token = Password::getRepository()->create($user);
            $data['url'] = config('app.url').'/verify-email/'.$user->id.'/'.$token;
            $data['user'] = $user;
            Mail::to($user->email)->send(new VerifyMail($data));

            Flash::success(__('messages.placeholder.registered_success'));

            return redirect(route('login'));
        } catch (\Exception $e) {
            DB::rollBack();

            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }
}
