<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Models\Subscription;
use App\Models\Transaction;
use Carbon\Carbon;
use App\Mail\PaymentSuccessMail;
use App\Repositories\SubscriptionRepository;
use App\Models\Plan;
use App\Models\CardRequest;

class OnePayController extends Controller
{

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(SubscriptionRepository $subscriptionRepository)
    {
        $this->subscriptionRepository = $subscriptionRepository;
    }

     public function makeonepaypaymet(Request $request)
    {
        $plan = Plan::with('currency')->findOrFail($request->planId);

        $data = $this->subscriptionRepository->manageSubscription($request->all());
        if (! isset($data['plan'])) { // 0 amount plan or try to switch the plan if it is in trial mode
            // returning from here if the plan is free.
            if (isset($data['status']) && $data['status'] == true) {
                return $this->sendSuccess($data['subscriptionPlan']->name . ' ' . __('messages.subscription_pricing_plans.has_been_subscribed'));
            } else {
                if (isset($data['status']) && $data['status'] == false) {
                    return $this->sendError(__('messages.placeholder.cannot_switch_to_zero'));
                }
            }
        }

        $subscriptionsPricingPlan = $data['plan'];
        $subscription = $data['subscription'];

        // Retrieve and sanitize input parameters
        $firstname = Auth::user()->first_name ?? '';
        $lastname =  Auth::user()->last_name ?? '';
        $email =     Auth::user()->email ?? '';
        $contact =   Auth::user()->phone ?? '+947123456789';
        $pay = isset($subscription->plan->price) ? number_format((float)$subscription->plan->price, 2, '.', '') : ''; // Ensure amount is formatted to two decimal places
        $reference = "1234567898"; // Static reference with at least 10 characters

        // Define application credentials
        $app_id = env('ONEPAY_APPID');
        $hash_salt = env('ONEPAY_HASH_TOKEN');
        $app_token = env('ONEPAY_APP_TOKEN');

        // Generate the hash
        $hash_string = $app_id . "LKR" . $pay . $hash_salt;
        $hash_result = hash('sha256', $hash_string);

        // Build the request payload
        $request_payload = [
            "currency" => "LKR",
            "amount" => (float)$pay, // Ensure amount is a decimal value
            "app_id" => $app_id,
            "reference" => $reference,
            "customer_first_name" => $firstname,
            "customer_last_name" => $lastname,
            "customer_phone_number" => $contact,
            "customer_email" => $email,
            "transaction_redirect_url" => route('onepay.successon') . '?id=' . $subscription->id . '&cardrequests=' . $request->cardrequests,
            "hash" => $hash_result,
            "additional_data" => "sample"
        ];

        // Encode the payload to JSON
        $data = json_encode($request_payload, JSON_UNESCAPED_SLASHES);

        // Configure cURL for API request
        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.onepay.lk/v3/checkout/link/',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_HTTPHEADER => [
                'Authorization: ' . $app_token,
                'Content-Type: application/json'
            ],
        ]);
        $response = curl_exec($curl);
       
        $err = curl_error($curl);

        curl_close($curl);

        $result = json_decode($response, true);

        return response()->json(['link' => $result['data']['gateway']['redirect_url'], 'status' => 200]);
    }

   public function successon(Request $request)
{
    try {

        // 1. Get subscription ID from URL
        $subscriptionID = $request->query('id');
        $cardrequests = $request->query('cardrequests');

        if (!$subscriptionID) {
            return redirect()->route('payment.cancel.page')
                ->with('error', 'Subscription ID missing.');
        }

        // 2. Read raw JSON from OnePay
        $inputJSON = file_get_contents('php://input');
        $input = json_decode($inputJSON, true);

        Log::info("OnePay Callback Raw JSON: " . $inputJSON);
        Log::info("OnePay Callback Request Data: ", $request->all());

        if (!$input) {
            return redirect()->route('payment.cancel.page')
                ->with('error', 'Invalid JSON payload.');
        }

        $transactionId = $input['transaction_id'] ?? 'N/A';
        $status = $input['status'] ?? 'UNKNOWN';

        // 3. Check subscription
        $subscription = Subscription::find($subscriptionID);

        if (!$subscription) {
            return redirect()->route('payment.cancel.page')
                ->with('error', 'Subscription not found.');
        }

        // 4. Prepare transaction data
        $transactionDetails = [
            'subscription_id' => $subscriptionID,
            'transaction_id' => $transactionId,
            'onepay_status' => $status,
            'raw_response' => $input,
            'total_amount' => $subscription->plan->price,
        ];

        // 5. Call success payment handler
        $result = $this->handleSuccessfulPayment($subscriptionID, $transactionDetails, 'onepay');

        // SUCCESS REDIRECT
        $token = csrf_token();
        $redirectUrl = route('payment.success.page') .
            "?_token={$token}&transid={$transactionId}&subscriptionId={$subscriptionID}&cardrequests={$cardrequests}";

        return redirect()->away($redirectUrl);

    } catch (\Exception $e) {

        Log::error('OnePay Success Error: ' . $e->getMessage());

        // FAIL REDIRECT – Your request
        return view('sadmin.plans.payment.paymentcancel')
            ->with('error', 'Payment failed. Please try again.');
    }
}


    public function initiate(Request $request)
    {
        // Example subscription retrieval
        $subscription = Subscription::findOrFail($request->subscription_id);

        // OnePay config from .env
        $appId     = config('services.onepay.appid');
        $hashToken = config('services.onepay.hash_token');
        $appToken  = config('services.onepay.app_token');
        $currency  = config('services.onepay.currency', 'LKR');

        $tranId = strtoupper(Str::random(10));

        $data = [
            'appid'                  => $appId,
            'hashToken'              => $hashToken,
            'amount'                 => $subscription->plan->price,
            'orderReference'         => $tranId,
            'customerFirstName'      => Auth::user()->first_name,
            'customerLastName'       => Auth::user()->last_name,
            'customerPhoneNumber'    => Auth::user()->phone,
            'customerEmail'          => Auth::user()->email,
            'transactionRedirectUrl' => route('onepay.callback'),
            'additionalData'         => 'returndata',
            'apptoken'                => $appToken,
            'currency'               => $currency,
        ];

        return response()->json([
            'status' => true,
            'onePayData' => $data
        ]);
    }

    public function callback(Request $request)
    {
        Log::info('OnePay callback received', $request->all());

        // The OnePay success payload
        $successData = $request->all();

        if (($successData['status'] ?? '') === 'SUCCESS') {
            try {
                $subscriptionId = $request->get('subscription_id');
                $subscription   = Subscription::findOrFail($subscriptionId);

                // Activate subscription
                $subscription->update([
                    'status'         => Subscription::ACTIVE,
                    'transaction_id' => $successData['transactionId'] ?? null
                ]);

                // Create transaction record
                $transaction = Transaction::create([
                    'tenant_id'      => $subscription->tenant_id,
                    'transaction_id' => $successData['transactionId'] ?? null,
                    'type'           => 'onepay',
                    'amount'         => $successData['amount'] ?? 0,
                    'status'         => Subscription::ACTIVE,
                    'meta'           => $successData
                ]);

                // Send email
                $user = Auth::user();
                Mail::to($user->email)->send(
                    new PaymentSuccessMail($subscription, $transaction->amount, $user)
                );

                return view('sadmin.plans.payment.paymentSuccess', [
                    'amountToPay' => $transaction->amount,
                    'formattedDate' => Carbon::parse($transaction->created_at)->format('Y-m-d H:i:s'),
                    'transactionIdFormatted' => 'ONEPAY-' . Carbon::now()->format('Ymd') . '-' . $transaction->id,
                    'plan_name' => $subscription->plan->name,
                    'subscriptionID' => $subscription->id
                ]);
            } catch (\Exception $e) {
                Log::error('OnePay success handler error: ' . $e->getMessage());
                return view('sadmin.plans.payment.paymentcancel');
            }
        }

        return view('sadmin.plans.payment.paymentcancel');
    }


    // public function onepaySuccess(Request $request)
    // {
    //     $data = $this->subscriptionRepository->manageSubscription($request->all());
    //     $subscription = $data['subscription'];

    //      return response()->json([
    //         'success' => $request->all(),
    //         // 'transaction_id' => $transactionDetails['transaction_id'] ?? null,

    //     ]);


    //     ($data);
    //     // $transactionDetails = $request->all();

    //     // // ✅ Reuse the same logic as PayWay
    //     // $subscriptionID = $transactionDetails['subscriptionId'] ?? null;
    //     // $result = $this->handleSuccessfulPayment($subscriptionID, $transactionDetails, 'onepay');


    // }
    public function createPendingOnePaySubscription(Request $request)
    {
        $plan = Plan::findOrFail($request->planId);
        // dd($plan);
        // Create a PENDING subscription
        $subscription = Subscription::create([
            'plan_id' => $plan->id,
            'plan_amount' => $plan->price,
            'tenant_id' => Auth::user()->tenant_id,
            'status' => Subscription::PENDING,
            'starts_at' => now(),
            'ends_at' => now()->addDays($plan->trial_days ?? 30),
        ]);

        return response()->json([
            'success' => true,
            'subscriptionId' => $subscription->id,
        ]);
    }

    public function showSuccessPage(Request $request)
    {

        $input = $request->all();
        $req_time = now()->utc()->format('YmdHis');

        $tran_id = $request->query('transid');
        $subscription = Subscription::where('transaction_id', $tran_id)->first();
        $subscriptionID = $subscription->id; // ✅ this is your primary key (id)
        $amountToPay = $subscription->payable_amount;
        $date = $subscription->created_at;
        $formattedDate = $date->format('Y-m-d H:i:s'); // Optional formatting
        $date = Carbon::parse($subscription->created_at)->format('Y-m-d');
        $existing = Transaction::where('transaction_id', $subscription->transaction_id)->first();

        if (!$existing && $transactionId) {
            $transaction = Transaction::create([
                'tenant_id' => $subscription->tenant_id,
                'transaction_id' => $transactionId,
                'type' => $paymentType,
                'amount' => $amountToPay,
                'status' => Subscription::ACTIVE,
                'meta' => $transactionDetails,
            ]);

            $date = $transaction->created_at;
            $formattedDate = $date->format('Y-m-d H:i:s'); // Optional formatting
        } else {
            // Already exists
            $transaction = $existing;
            // dd($transaction);
            $formattedDate = $transaction->created_at->format('Y-m-d H:i:s');
        }
        $prefix = 'ODR';
        $plan = \App\Models\Plan::find($subscription->plan_id);
        $plan_name = $plan->name;

        if (!empty($input['cardrequests'])) {

            // 1. Get subscription_id from CardRequest
            $cardRequest = \App\Models\CardRequest::find($input['cardrequests']);

            if ($cardRequest) {
                $subscriptionId = $cardRequest->subscription_id;

                // 2. Get per_card_price from Plans table (plan id = 4)


                if ($plan) {
                    $prefix = 'ODR-C';
                    Subscription::findOrFail($subscriptionID)->update(['no_of_vcards' => $subscriptionId]);
                    $perCardPrice = $plan->per_card_price;

                    // 3. Calculate price
                    $totalPrice = $perCardPrice * $subscriptionId;

                    // 4. Update CardRequest (or wherever you're updating)
                    $plan->price = $totalPrice;
                    $plan->no_of_vcards = $subscriptionId;
                    $plan->save();
                }
            }
        }

        $date = Carbon::parse($transaction->created_at)->format('Y-m-d');


        $transactionIdFormatted = $prefix . '-' . $date . '-' . $transaction->id;
        
        $paymentMethod = $subscription->payment_type;

        // Get the response
        return view('sadmin.plans.payment.paymentSuccess', compact(
            'amountToPay',
            'formattedDate',
            'transactionIdFormatted',
            'plan_name',
            'subscriptionID',
            'paymentMethod'
        ));
    }


    public function onepaySuccess(Request $request)
    {

        try {
            // subscription data from repository
            $data = $this->subscriptionRepository->manageSubscription($request->all());
            $tran_id = $request->input('transaction_id');

            // Step 1: Find the subscription using the transaction ID
            $subscription = Subscription::where('transaction_id', $tran_id)->first();
            $subscriptionID = $subscription->id;
            // dd($subscriptionID);
            if (!$subscriptionID) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subscription ID not found',
                ], 400);
            }

            // transaction details (successData from frontend body)
            $transactionDetails = $request->all();

            // ✅ call the reusable success handler
            $result = $this->handleSuccessfulPayment($subscriptionID, $transactionDetails, 'onepay');

            return response()->json($result);
        } catch (\Exception $e) {
            Log::error('OnePay Success Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'OnePay Success handler failed',
            ], 500);
        }
    }



    public function handleSuccessfulPayment($subscriptionID, $transactionDetails, $paymentType = 'onepay')
    {
        try {
            $amountToPay = $transactionDetails['data']['total_amount'];
            $subscription = Subscription::findOrFail($subscriptionID);

            // Activate current subscription
            $subscription->update([
                'status' => Subscription::ACTIVE,
                'transaction_id' => $transactionDetails['transaction_id'],
                'amount' => $amountToPay,
                'payment_type' => 'Onepay',
            ]);


            // Create transaction record if not exists
            $transactionId = $transactionDetails['transaction_id'];
            $transaction = Transaction::firstOrCreate(
                ['transaction_id' => $transactionId],
                [
                    'tenant_id' => $subscription->tenant_id,
                    'type' => $paymentType,
                    'amount' => $amountToPay,
                    'status' => Subscription::ACTIVE,
                    'meta' => $transactionDetails,
                ]
            );

            $formattedDate = $transaction->created_at->format('Y-m-d H:i:s');
            $prefix = 'ODR';

            $plan = Plan::find($subscription->plan_id);
            $plan_name = $plan->name ?? 'N/A';

            // Handle CardRequest if present
            if (!empty($transactionDetails['data']['cardrequests'])) {
                $cardRequest = CardRequest::find($transactionDetails['data']['cardrequests']);
                if ($cardRequest && $plan) {
                    $prefix = 'ODR-C';
                    $subscriptionId = $cardRequest->subscription_id;
                    $perCardPrice = $plan->per_card_price;
                    $totalPrice = $perCardPrice * $subscriptionId;

                    $plan->update([
                        'price' => $totalPrice,
                        'no_of_vcards' => $subscriptionId,
                    ]);
                }
            } else {
                // Deactivate others
                Subscription::whereTenantId($subscription->tenant_id)
                    ->where('id', '!=', $subscriptionID)
                    ->where('status', '!=', Subscription::REJECT)
                    ->update(['status' => Subscription::INACTIVE]);
            }

            $transactionIdFormatted = $prefix . '-' . now()->format('Y-m-d') . '-' . $transaction->id;

            // Send success mail
            $data = $transactionDetails['data'] ?? [];
            $email = $data['email'] ?? null;
            $firstName = $data['first_name'] ?? '';
            $lastName = $data['last_name'] ?? '';
            $name = trim("$firstName $lastName");

            // if ($email) {
            //     $pseudoUser = (object)[
            //         'first_name' => $firstName,
            //         'last_name' => $lastName,
            //         'name' => $name,
            //         'email' => $email,
            //         'transaction_id' => $transactionId,
            //     ];
            //     Mail::to($email)->send(new PaymentSuccessMail($subscription, $amountToPay, $pseudoUser));
            // }

            return [
                'success' => true,
                'amountToPay' => $amountToPay,
                'formattedDate' => $formattedDate,
                'transactionIdFormatted' => $transactionIdFormatted,
                'plan_name' => $plan_name,
                'subscriptionID' => $subscriptionID,
            ];
        } catch (\Exception $e) {
            \Log::error('OnePay Success faild: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'OnePay Success handler failed',
                'error'   => $e->getMessage(), // ✅ send error details
            ], 500);
        }
    }






    public function fail(Request $request)
    {
        Log::warning('OnePay payment failed', $request->all());
        return view('sadmin.plans.payment.paymentcancel');
    }
}
